// HIDAcme.cpp: implementation of the HIDAcme class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "HIDAcme.h"
#include "HIDDevice.h"
#include "HIDSocket.h"
#include "HIDDescriptor.h"
#include "HIDGraphics.h"
#include "HEXFile.h"

#define A_TIMEOUT		3000
#define BUF_LINE_LEN	47
#define STR_LINE_LEN	57
//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

HIDAcme::HIDAcme(HIDDevice* hdev, int timeout, int ir_decoder)
:HIDInterface(hdev)
,m_irDecoder(ir_decoder)
{
	if (hdev)
		MyTrace(0, "HID device %s %s opened\n"
		, hdev->GetDescriptor()->m_sManufacturer
		, hdev->GetDescriptor()->m_sProduct);
	else MyTrace(0, "HID device could not be opened\n");
		
	m_socket = NULL;
	m_nError = -1;
	m_bWriteOK = true;
	m_pGraphics = NULL;

	m_nTimeout = timeout;
	
	int cnt = 0;
	do{
		if (!isFlasher())
		{
			setLedStatus(0);
			setLCDControl(4);
			setBacklight(1);
			setContrast(22);
			

			setText(0,0, "    USB-LCD-2x20    ");
			setText(0,1, "  www.mini-box.com  ");
		}
		else
		{
			getVer();
			getVer();
			getVer();
			getVer();
			getVer();
		}

		if (!m_bWriteOK)
		{
			CloseDevice();
			Sleep(1000);
			OpenDevice();
		}
		cnt++;
	}while ((!m_bWriteOK)&&(cnt<5));
}

HIDAcme::~HIDAcme()
{
	m_nError = 0;
}

void HIDAcme::SetSocket(HIDSocket* sock)
{
	m_lockSock.lock();
	m_socket = sock;
	m_lockSock.unlock();
}

void HIDAcme::Send2Socket(const char* mesg, ...)
{
	m_lockSock.lock();
	if (m_socket)
	{
		va_list args;
		va_start(args, mesg);

		char szBuffer[MAX_BUF+1];
#ifdef _WIN32
		_vsnprintf(szBuffer, MAX_BUF, mesg, args);
#else
		vsnprintf(szBuffer, MAX_BUF, mesg, args);
#endif
		szBuffer[MAX_BUF] = 0;
		
		m_socket->send2all(szBuffer);
	}
	m_lockSock.unlock();
}

void HIDAcme::Received(HArray* array)
{
	m_aReceived.setValue(array);

	switch (array->getBuf()[0])
	{
	case IN_REPORT_KEY_STATE:
		{
			MyTrace(3, ">>> KP %x %x\n", array->getBuf()[1], array->getBuf()[2]);

			Send2Socket("notify key x%02x x%02x", array->getBuf()[1], array->getBuf()[2]);
		}break;
	case IN_REPORT_IR_DATA:
		{
			MyTrace(3, ">>> IR (%02d)", array->getBuf()[1]);
			int i = 0;
			for (i=0;i<array->getBuf()[1];i++)
				MyTrace(2, " %x", array->getBuf()[2+i]);
			MyTrace(2, "\n");

			if (m_irDecoder.getDecodeType() == IR_RC5)
			{
				int result = m_irDecoder.decodeIR(array->getBuf()+2, array->getBuf()[1]);
				if (result != -1)
				{
					char buf[256];
					sprintf(buf, "notify ir_rc5 x%04x",(unsigned short)(result&0xFFFF));
					Send2Socket(buf);
				}
			}
			else
			{
				char buf[256], bt[5];
				sprintf(buf, "notify ir %d",array->getBuf()[1]);
				for (i=0;i<array->getBuf()[1];i++)
				{
					sprintf(bt," x%02x", array->getBuf()[2+i]);
					strcat(buf, bt);
				}
				
				Send2Socket(buf);
			}
		}break;
	case RESULT_PARAMETER_MISSING:
		{
			MyTrace(2, "Parameter missing\n");
		}break;
	case RESULT_DATA_MISSING:
		{
			MyTrace(2, "Data missing\n");
		}break;
	case RESULT_BLOCK_READ_ONLY:
		{
			MyTrace(2, "Block readonly\n");
		}break;
	case RESULT_BLOCK_TOO_BIG:
		{
			MyTrace(2, "Block too big\n");
		}break;
	case RESULT_SECTION_OVERFLOW:
		{
			MyTrace(2, "Section overflow\n");
		}break;
	case HID_REPORT_GET_VERSION:
		{
			MyTrace(2, "HID_REPORT_GET_VERSION\n");
			m_nError = 0;
		}break;
	case HID_REPORT_ERASE_MEMORY:
		{
			MyTrace(2, "HID_REPORT_ERASE_MEMORY\n");
			m_nError = 0;
		}break;
	case HID_REPORT_READ_MEMORY:
		{
			MyTrace(2, "HID_REPORT_READ_MEMORY\n");
			m_nError = 0;
		}break;
	case HID_REPORT_WRITE_MEMORY:
		{
			MyTrace(2, "HID_REPORT_WRITE_MEMORY\n");
			m_nError = 0;
		}break;
	case IN_REPORT_EXT_EE_DATA:
		{
			MyTrace(2, "IN_REPORT_EXT_EE_DATA\n");
			m_nError = 0;
		}break;
	case IN_REPORT_INT_EE_DATA:
		{
			MyTrace(2, "IN_REPORT_INT_EE_DATA\n");
			m_nError = 0;
		}break;
	case OUT_REPORT_RELAY_ONOFF:
		{
			MyTrace(2, "OUT_REPORT_RELAY_ONOFF\n");
			m_nError = 0;
		}break;
/*	case OUT_REPORT_EXT_EE_READ:
		{
			MyTrace(2, "OUT_REPORT_EXT_EE_READ\n");
			m_nError = 0;
		}break;
	case OUT_REPORT_EXT_EE_WRITE:
		{
			MyTrace(2, "OUT_REPORT_EXT_EE_WRITE\n");
			m_nError = 0;
		}break;
	case OUT_REPORT_INT_EE_READ:
		{
			MyTrace(2, "OUT_REPORT_INT_EE_READ\n");
			m_nError = 0;
		}break;
	case OUT_REPORT_INT_EE_WRITE:
		{
			MyTrace(2, "OUT_REPORT_INT_EE_WRITE\n");
			m_nError = 0;
		}break;*/
	case HID_REPORT_EXIT_FLASHER://ff is errorcode as well in this case
		{
			MyTrace(2, "HID_REPORT_EXIT_FLASHER\n");
			if (array->length() > 1)
				 m_nError = array->getBuf()[1];
			else m_nError = 2;
		}break;
	case HID_REPORT_EXIT_KEYBOARD://ff is errorcode as well in this case
		{
			MyTrace(2, "HID_REPORT_EXIT_KEYBOARD\n");
			if (array->length() > 1)
				 m_nError = array->getBuf()[1];
			else m_nError = 2;
		}break;
	default:
		{
			MyTrace(0, "Unknown income %x\n", array->getBuf()[0]);
			m_nError = 1;
		}
	}
}

void HIDAcme::getVer()
{
	if (!m_pParent) return;

	HArray ha;
	ha.setHIDPkt(HID_REPORT_GET_VERSION, 0, 0);
	m_bWriteOK = m_pParent->Write(&ha);
}

void HIDAcme::setLedStatus(unsigned char status)
{
	m_cLedStatus = status;
	Send2Socket("set led x%02x", m_cLedStatus);

	if (!m_pParent) return;

	HArray ha;
	ha.setHIDPkt(OUT_REPORT_LED_STATE, 1, 1, status);
	m_bWriteOK = m_pParent->Write(&ha);
}

void HIDAcme::setLCDControl(unsigned char status)
{
	m_cLCDStatus = status;
	Send2Socket("set lcd x%02x", m_cLCDStatus);

	if (!m_pParent) return;

	HArray ha;
	ha.setHIDPkt(OUT_REPORT_LCD_CONTROL, 1, 1, status);
	m_bWriteOK = m_pParent->Write(&ha);
}

void HIDAcme::setBacklight(unsigned char status)
{
	m_cBacklight = status;
	Send2Socket("set blight %d", m_cBacklight);

	if (!m_pParent) return;

	HArray ha;
	ha.setHIDPkt(OUT_REPORT_LCD_BACKLIGHT, 1, 1, status);
	m_bWriteOK = m_pParent->Write(&ha);
}

void HIDAcme::setContrast(unsigned char status)
{
	m_cContrast = status;
	Send2Socket("set contrast %d", m_cContrast);

	if (!m_pParent) return;

	HArray ha;
	ha.setHIDPkt(OUT_REPORT_LCD_CONTRAST, 1, 1, status);
	m_bWriteOK = m_pParent->Write(&ha);
}

void HIDAcme::setText(const char x, const char y, const char* text, unsigned int len)
{
	if (len == 0) len = strlen(text);
	if (len > LINE_LENGTH) len = LINE_LENGTH;

	HArray ha;
	ha.setHIDPkt(OUT_REPORT_LCD_TEXT, 3+len, 3, y, x, (unsigned char)len);

	int nx = x;
	char* line = m_cLine1;
	if (y != 0) line = m_cLine2;

	for (unsigned int i=0;i<len;i++)
	{
		ha.getBuf()[4+i] = text[i];
		
		if (nx < LINE_LENGTH) line[nx++] = text[i];
	}

	char buf[LINE_LENGTH*2+1];
	GetLine(buf, y);
	Send2Socket("set text 0 %d %s",y, buf);

	if (!m_pParent) return;

	m_bWriteOK = m_pParent->Write(&ha);
}

void HIDAcme::setRelay(unsigned long msec)
{
	if (!m_pParent) return;

	HArray ha;
	ha.setHIDPkt(OUT_REPORT_RELAY_ONOFF, 2, 2, msec & 0xff, (msec >> 8) & 0xff);
	m_bWriteOK = m_pParent->Write(&ha);
}

void HIDAcme::testSplash()
{
	char* text = "--- splash test --- ";
	for (unsigned int i=0;i<LINE_LENGTH;i++)
	{
		m_cLine1[i] = text[i];
		m_cLine2[i] = text[i];
	}

	Send2Socket("set splash");

	if (!m_pParent) return;

	HArray ha;
	ha.setHIDPkt(OUT_REPORT_TESTSPLASH, 1, 1, 0xff);
	m_bWriteOK = m_pParent->Write(&ha);
}


void HIDAcme::GetLine(char* buf, int ln)
{
	char* line = m_cLine1;
	if (ln != 0) line = m_cLine2;

	int cnt_buf = 0;
	for (unsigned int n=0;n<LINE_LENGTH;n++)
	{
		switch (line[n])
		{
		case '\\'	:buf[cnt_buf++] = '\\';buf[cnt_buf++] = '\\';break;
		case 0		:buf[cnt_buf++] = '\\';buf[cnt_buf++] = '0';break;
		case 1		:buf[cnt_buf++] = '\\';buf[cnt_buf++] = '1';break;
		case 2		:buf[cnt_buf++] = '\\';buf[cnt_buf++] = '2';break;
		case 3		:buf[cnt_buf++] = '\\';buf[cnt_buf++] = '3';break;
		case 4		:buf[cnt_buf++] = '\\';buf[cnt_buf++] = '4';break;
		case 5		:buf[cnt_buf++] = '\\';buf[cnt_buf++] = '5';break;
		case 6		:buf[cnt_buf++] = '\\';buf[cnt_buf++] = '6';break;
		case 7		:buf[cnt_buf++] = '\\';buf[cnt_buf++] = '7';break;
		case 8		:buf[cnt_buf++] = '\\';buf[cnt_buf++] = '8';break;
		case 9		:buf[cnt_buf++] = '\\';buf[cnt_buf++] = '9';break;
		default		:buf[cnt_buf++] = line[n];
		}
		buf[cnt_buf] = 0;
	}
}

void HIDAcme::getinitmesgs(char* buf)
{
	char buf1[LINE_LENGTH*2+1],buf2[LINE_LENGTH*2+1];
	GetLine(buf1, 0);
	GetLine(buf2, 1);

	sprintf(buf, "set led x%02x\r\nset lcd x%02x\r\nset blight %d\r\nset contrast %d\r\nset text 0 0 %s\r\nset text 0 1 %s"
		,m_cLedStatus,m_cLCDStatus,m_cBacklight,m_cContrast,buf1,buf2);
	
}

void HIDAcme::clearLCD()
{
	memset(m_cLine1, ' ', LINE_LENGTH); 
	memset(m_cLine2, ' ', LINE_LENGTH); 
	
	Send2Socket("set clear");


	if (!m_pParent) return;

	HArray ha;
	ha.setHIDPkt(OUT_REPORT_LCD_CLEAR, 0, 0);
	m_bWriteOK = m_pParent->Write(&ha);
}

void HIDAcme::setFont(const char n, const char map[8])
{
	if (!m_pParent) return;

	HArray ha;
	ha.setHIDPkt(OUT_REPORT_LCD_FONT, 1+8, 1+8, n, map[0], map[1], map[2], map[3], map[4], map[5], map[6], map[7]);

	m_bWriteOK = m_pParent->Write(&ha);
}

bool HIDAcme::isFlasher()
{
	if (!m_pParent) return false;
	return (m_pParent->GetDescriptor()->m_nPID == 0x01);
}

void HIDAcme::switchMode()
{
	MyTrace(0, "Switching mode, tout=%d(%x)\n",m_nTimeout,m_nTimeout);

//	HArray ha1;
//	ha1.setHIDPkt(HID_SET_SNOOZE_TIME, 2, 2, m_nTimeout & 0xff, (m_nTimeout >> 8) & 0xff);
//	m_bWriteOK = m_pParent->Write(&ha1);

	m_nError = -1;

	if (isFlasher())
	{
		HArray ha;
		ha.setHIDPkt(HID_REPORT_EXIT_FLASHER, 2, 2, m_nTimeout & 0xff, (m_nTimeout >> 8) & 0xff);
		m_bWriteOK = m_pParent->Write(&ha);
	}
	else
	{
		HArray ha;
		ha.setHIDPkt(HID_REPORT_EXIT_KEYBOARD, 2, 2, m_nTimeout & 0xff, (m_nTimeout >> 8) & 0xff);
		m_bWriteOK = m_pParent->Write(&ha);
	}

	int tmout = 0;
	while (m_nError == -1)
	{
		Sleep(5);
		tmout++;
		if (tmout > A_TIMEOUT) return;
	}
}

bool HIDAcme::_isEnabled(int type)
{
	switch (type)
	{
	case TYPE_CODE_MEMORY:
		{
			if (isFlasher()) return true;
		}break;
	case TYPE_EPROM_EXTERNAL:
		{
			if (!isFlasher()) return true;
		}break;
	case TYPE_EPROM_INTERNAL:
		{
			if (!isFlasher()) return true;
		}break;
	}
	return false;
}

int HIDAcme::_getWriteMsg(int type)
{
	switch (type)
	{
	case TYPE_CODE_MEMORY:		return HID_REPORT_WRITE_MEMORY;
	case TYPE_EPROM_EXTERNAL:	return OUT_REPORT_EXT_EE_WRITE;
	case TYPE_EPROM_INTERNAL:	return OUT_REPORT_INT_EE_WRITE;
	}
	return 0;
}

int HIDAcme::_getReadMsg(int type)
{
	switch (type)
	{
	case TYPE_CODE_MEMORY:		return HID_REPORT_READ_MEMORY;
	case TYPE_EPROM_EXTERNAL:	return OUT_REPORT_EXT_EE_READ;
	case TYPE_EPROM_INTERNAL:	return OUT_REPORT_INT_EE_READ;
	}
	return 0;
}

bool HIDAcme::writeFile(bool fromsocket, int type, const char* file, const unsigned long end)
{
	if (!_isEnabled(type)) return false;

	m_lockCommand.lock();
	int n = _writeFile(fromsocket, type, file, end);
	if (n != 0)
	{
		if (type == TYPE_CODE_MEMORY)
		{
			MyTrace(0, "Error occured writing file: %d > ERASING address x2000\n", n);
			
			unsigned long addr = 0x2000;

			HArray erase;
			erase.setHIDPkt(HID_REPORT_ERASE_MEMORY, 3, 3, addr & 0xff, (addr >> 8) & 0xff, (addr >> 16) & 0xff);
			m_bWriteOK = m_pParent->Write(&erase);
		}
		else MyTrace(0, "Error occured writing file: %d (type=%d)\n", n, type);

		if (m_pGraphics)
			m_pGraphics->UpdateStatus(fromsocket, STATUS_ERROR, STATUS_WRITE, n);
	}
	else
	{
		MyTrace(0, "Writing file succesfull: %d (type=%d)\n", n, type);

		if (m_pGraphics)
			m_pGraphics->UpdateStatus(fromsocket, STATUS_OK, STATUS_WRITE, 0);
	}
	m_lockCommand.unlock();

	return (n==0);
}

int HIDAcme::_writeFile(bool fromsocket, int type, const char* file, const unsigned long end)
{
	int tmout = 0;
	HEXFile hfile(file);
	if (hfile.Open() == 0)
	{
		unsigned long addr,oaddr = -1;
		int ret;

		if (type == TYPE_CODE_MEMORY)
		{
			oaddr = -1;
			ret = hfile.NextLine();
			while (ret==0)
			{
				if (m_pGraphics)
					m_pGraphics->UpdateStatus(fromsocket, STATUS_ERASE, hfile.GetMaxLines(), hfile.GetCurLine()+1);

				addr = hfile.GetGlobalAddress();

				if (end != -1)
				{
					if (addr >= end)
					{
						MyTrace(1, "erase: end reached 1(%x)\n",addr);
						addr = end - 64;
						
						ret = -1000;
					}
					else
					{
						if ((addr + 64) >= end)
						{
							MyTrace(1, "erase: end reached 2(%x > %x)\n",addr,end - 64);

							addr = end - 64;
							ret = -1000;
						}
					}
				}
				
				if ((oaddr == -1)||((addr-oaddr)>63))
				{
					m_nError = -1;

					HArray erase;
					erase.setHIDPkt(HID_REPORT_ERASE_MEMORY, 3, 3, addr & 0xff, (addr >> 8) & 0xff, (addr >> 16) & 0xff);
					m_bWriteOK = m_pParent->Write(&erase);

					tmout = 0;
					while (m_nError == -1)
					{
						Sleep(5);
						tmout++;
						if (tmout > A_TIMEOUT) return -1;
					}

					if (m_nError != 0) return m_nError;

					MyTrace(1, "erase: addr: %x (%d)\n",addr, hfile.GetLength());

					oaddr = addr;
				}
				else MyTrace(1, " jump: addr: %x (%d)\n",addr, hfile.GetLength());

				if (ret != -1000)
				{//last line reached end
					ret = hfile.NextLine();
					if (ret > 1) return -2;
				}
			}

			HArray erase;
			erase.setHIDPkt(HID_REPORT_ERASE_MEMORY, 3, 3, addr & 0xff, (addr >> 8) & 0xff, (addr >> 16) & 0xff);
			m_bWriteOK = m_pParent->Write(&erase);

			tmout = 0;
			while (m_nError == -1)
			{
				Sleep(5);
				tmout++;
				if (tmout > A_TIMEOUT) return -3;
			}

			Sleep(50);

			if (m_nError != 0) return m_nError;
			MyTrace(0, "erase: addr: %x (%d)\n",addr, hfile.GetLength());
		}

		hfile.Reset();
		ret = hfile.NextLine();
		int len;
		while (ret==0)
		{
			if (m_pGraphics)
				m_pGraphics->UpdateStatus(fromsocket, STATUS_WRITE, hfile.GetMaxLines(), hfile.GetCurLine()+1);

			addr = hfile.GetGlobalAddress();
			len  = hfile.GetLength();

			if (end != -1)
			{
				if (addr >= end)
				{
					MyTrace(0, "write: jump1: %x \n",addr);
					break;
				}

				if ((addr + len) >= end)
				{
					len = end - addr;
					ret = -1000;

					MyTrace(0, "write: jump2: %x (len=%d) \n",addr, len);
				}
			}

			m_nError = -1;
			
			HArray ha;

			if (type == TYPE_CODE_MEMORY)
			{
				ha.setHIDPkt(_getWriteMsg(type), 4+len, 4, addr & 0xff, (addr >> 8) & 0xff, (addr >> 16) & 0xff, len);
	 			for (int i=0;i<len;i++)
					ha.getBuf()[5+i] = hfile.GetData()[i];
			}
			else
			{
				ha.setHIDPkt(_getWriteMsg(type), 3+len, 3, addr & 0xff, (addr >> 8) & 0xff, len);
				for (int i=0;i<len;i++)
					ha.getBuf()[4+i] = hfile.GetData()[i];
			}

				
			m_bWriteOK = m_pParent->Write(&ha);
			tmout = 0;
			while (m_nError == -1)
			{
				Sleep(5);
				tmout++;
				if (tmout > A_TIMEOUT) return -14;
			}
			if (m_nError != 0) return m_nError;

			if (type == TYPE_CODE_MEMORY)
			{
				if (!ha.equals(&m_aReceived, 4+len))
					return -3;
			}
			else
			{
				if (!ha.equalsSpecial(&m_aReceived, 3+len))
					return -3;
			}

			MyTrace(1, "write: addr: %x (%d)\n",addr, len);

			if (ret != -1000)
			{
				ret = hfile.NextLine();
				if (ret > 1) return -15;
			}
		}
	}
	else
	{
		MyTrace(1,"Invalid hex file!\n");

		return -100;
	}
	
	return 0;
}

void HIDAcme::eraseFlash(bool fromsocket, const unsigned long start, const unsigned long end)
{
	if (!isFlasher()) return;

	m_lockCommand.lock();
	int n = _eraseFlash(fromsocket, start, end);
	if (n != 0)
	{
		MyTrace(0, "Error occured erasing: %d\n", n);
		
		if (m_pGraphics)
			m_pGraphics->UpdateStatus(fromsocket, STATUS_ERROR, STATUS_ERASE, n);
	}
	else
	{
		MyTrace(0, "Erasing succesfull: %d\n", n);

		if (m_pGraphics)
			m_pGraphics->UpdateStatus(fromsocket, STATUS_OK, STATUS_ERASE, 0);
	}
	m_lockCommand.unlock();
}

int HIDAcme::_eraseFlash(bool fromsocket, const unsigned long start, const unsigned long end)
{
	if (end<=start) return -100;

	unsigned long lines = (end - start) / 64;

	int tmout;
	unsigned long addr = start;
	while (addr < end)
	{
		if (m_pGraphics)
			m_pGraphics->UpdateStatus(fromsocket, STATUS_ERASE, lines, ((addr - start) / 64) + 1);

		m_nError = -1;

		HArray erase;
		erase.setHIDPkt(HID_REPORT_ERASE_MEMORY, 3, 3, addr & 0xff, (addr >> 8) & 0xff, (addr >> 16) & 0xff);
		m_bWriteOK = m_pParent->Write(&erase);

		tmout = 0;
		while (m_nError == -1)
		{
			Sleep(5);
			tmout++;
			if (tmout > A_TIMEOUT) return -1;
		}

		if (m_nError != 0) return m_nError;

		addr += 64;
	}

	return 0;
}

bool HIDAcme::readFile(bool fromsocket, int type, const char* file,const unsigned long start, const unsigned long end)
{
	if (!_isEnabled(type)) return false;

	m_lockCommand.lock();
	int n = _readFile(fromsocket, type, file, start, end);
	if (n != 0)
	{
		MyTrace(0, "Error occured reading: %d (type=%d)\n", n, type);
		
		if (m_pGraphics)
			m_pGraphics->UpdateStatus(fromsocket, STATUS_ERROR, STATUS_READ, n);
	}
	else
	{
		MyTrace(0, "Reading succesfull: %d (type=%d)\n", n, type);

		if (m_pGraphics)
			m_pGraphics->UpdateStatus(fromsocket, STATUS_OK, STATUS_READ, 0);
	}
	m_lockCommand.unlock();

	return (n==0);
}

int HIDAcme::_readFile(bool fromsocket, int type, const char* file,const unsigned long start, const unsigned long end)
{
	if (end<=start) return -100;

	unsigned long lines = (end - start) / 32;

	FILE* fw = fopen(file, "wt");
	if (!fw) return -1;

	fprintf(fw, ":020000040000FA\r\n");

	int tmout;
	unsigned long addr = start, oldaddr = start;
	while (addr < end)
	{
		if (m_pGraphics)
			m_pGraphics->UpdateStatus(fromsocket, STATUS_READ, lines, ((addr - start) / 32) + 1);

		m_nError = -1;

		HArray read;
		if (type == TYPE_CODE_MEMORY)
			 read.setHIDPkt(_getReadMsg(type), 4, 4, addr & 0xff, (addr >> 8) & 0xff, (addr >> 16) & 0xff, 32);
		else read.setHIDPkt(_getReadMsg(type), 3, 3, addr & 0xff, (addr >> 8) & 0xff, 20);

		m_bWriteOK = m_pParent->Write(&read);

		tmout = 0;
		while (m_nError == -1)
		{
			Sleep(5);
			tmout++;
			if (tmout > A_TIMEOUT)
			{
				fprintf(fw, "ERROR: timeout\r\n");
				fclose(fw);
				return -2;
			}
		}

		if (m_nError != 0)
		{
			fprintf(fw, "ERROR: device responded with %x\r\n", m_nError);
			fclose(fw);
			return m_nError;
		}
/*
TX[37]:F3 A0 23 00 20 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 0
0 00 00 00 00 00 00 00 00 00 00 00 00
RX[37]:F3 A0 23 00 20 FF FF FF FF FF FF FF FF FF FF FF FF FF FF FF FF FF FF FF F
F FF FF FF FF FF FF FF FF FF FF FF FF
*/

		unsigned char checksum = 0;
		if (type == TYPE_CODE_MEMORY)
		{
			fprintf(fw, ":%02X",m_aReceived.getAt(4));
			checksum += m_aReceived.getAt(4);
			fprintf(fw, "%02X",((addr >> 8) & 0xff));
			checksum += ((addr >> 8) & 0xff);
			fprintf(fw, "%02X",addr & 0xff);
			checksum += (addr & 0xff);
			fprintf(fw, "00");
			for (unsigned int i=5;i<m_aReceived.length();i++)
			{
				fprintf(fw, "%02X",m_aReceived.getAt(i));
				checksum += m_aReceived.getAt(i);
			}
			checksum = 0x01 + ~checksum;
			fprintf(fw, "%02X\r\n",checksum);

			addr += 32;
		}
		else
		{
			fprintf(fw, ":%02X",m_aReceived.getAt(3));
			checksum += m_aReceived.getAt(3);
			fprintf(fw, "%02X",((addr >> 8) & 0xff));
			checksum += ((addr >> 8) & 0xff);
			fprintf(fw, "%02X",addr & 0xff);
			checksum += (addr & 0xff);
			fprintf(fw, "00");
			for (unsigned int i=4;i<m_aReceived.length();i++)
			{
				fprintf(fw, "%02X",m_aReceived.getAt(i));
				checksum += m_aReceived.getAt(i);
			}
			checksum = 0x01 + ~checksum;
			fprintf(fw, "%02X\r\n",checksum);	
			
			addr += 20;
		}
	}

	fprintf(fw,":00000001FF");
	fclose(fw);
	return 0;
}

bool HIDAcme::writeFile4String(const char* file, const char* str)
{
	unsigned char buffer[5*BUF_LINE_LEN];

	char tstr[5*STR_LINE_LEN];
	memset(tstr, 0, 5*STR_LINE_LEN);
	strncpy(tstr, str, 5*STR_LINE_LEN);

	char no[3];no[2] = 0;
	unsigned long nTime = 0;
	int m,s;
	long led,nextslot,rep;

	for (int i=0;i<5;i++)
	{
		no[0] = tstr[i*STR_LINE_LEN];no[1] = tstr[i*STR_LINE_LEN+1];
		m = atoi(no);if (m>59) m = 59;

		no[0] = tstr[i*STR_LINE_LEN+3];no[1] = tstr[i*STR_LINE_LEN+4];
		s = atoi(no);if (s>59) s = 59;

		if ((m <= -1) || (s <= -1))
			 nTime = 0xffff;
		else nTime = m * 60 + s;

		buffer[i*BUF_LINE_LEN]      = (char) (nTime & 0xff);
		buffer[i*BUF_LINE_LEN+1]	= (char) ((nTime >> 8) & 0xff);

		no[0] = tstr[i*STR_LINE_LEN+6];no[1] = tstr[i*STR_LINE_LEN+7];
		nextslot = strtol(no, NULL, 16);
		buffer[i*BUF_LINE_LEN+2]	= (char) (nextslot & 0xff);

		no[0] = tstr[i*STR_LINE_LEN+9];no[1] = tstr[i*STR_LINE_LEN+10];
		rep = strtol(no, NULL, 16);
		buffer[i*BUF_LINE_LEN+3]	= (char) (rep & 0xff);

		no[0] = tstr[i*STR_LINE_LEN+12];no[1] = tstr[i*STR_LINE_LEN+13];
		led = strtol(no, NULL, 16);
		buffer[i*BUF_LINE_LEN+4]	= (char) (led & 0xff);

		int j = 0;
		for (j=0;j<20;j++)
		{//first row
			buffer[i*BUF_LINE_LEN+5+j] = tstr[i*STR_LINE_LEN+15+j];
			if (buffer[i*BUF_LINE_LEN+5+j] == 0) buffer[i*BUF_LINE_LEN+5+j] = ' ';
		}

		buffer[i*BUF_LINE_LEN+25] = 0;

		for (j=0;j<20;j++)
		{//second row
			buffer[i*BUF_LINE_LEN+26+j] = tstr[i*STR_LINE_LEN+36+j];
			if (buffer[i*BUF_LINE_LEN+26+j] == 0) buffer[i*BUF_LINE_LEN+26+j] = ' ';
		}

		buffer[i*BUF_LINE_LEN+46] = 0;
	}

	return writeFile4Array(file, buffer, 5*BUF_LINE_LEN);
}

bool HIDAcme::writeFile4Array(const char* file, const unsigned char* buffer, const unsigned long length)
{
	FILE* fw = fopen(file, "wt");
	if (!fw) return false;

	fprintf(fw, ":020000040000FA\r\n");
	unsigned long addr = 0;
	unsigned char checksum = 0;
	int len;
	while (addr < length)
	{
		checksum = 0;
		
		len = 20;
		if ((addr + 20) > length) len = length - addr;


		fprintf(fw, ":%02X",len);
		checksum += len;
		fprintf(fw, "%02X",((addr >> 8) & 0xff));
		checksum += ((addr >> 8) & 0xff);
		fprintf(fw, "%02X",addr & 0xff);
		checksum += (addr & 0xff);
		fprintf(fw, "00");

		for (unsigned long i=addr;i<(addr+len);i++)
		{
			fprintf(fw, "%02X",(unsigned char)buffer[i]);
			checksum += buffer[i];
		}

		checksum = 0x01 + ~checksum;
		fprintf(fw, "%02X\r\n",checksum);

		addr+=20;
	}

	fprintf(fw,":00000001FF");

	fclose(fw);

	return true;
}

void HIDAcme::sendCmd(const char* command, bool from_socket)
{
	if (!command) return;
	int len = strlen(command);
	if (!len) return;

	m_lockCommand.lock();

	char* tcmd = new char[len+1];
	strcpy(tcmd, command);
	char* cmd = strtok(tcmd, " \r\n");
	char* sub = strtok(NULL, " \r\n");
	
	bool ok = false;
	if (strcmp(cmd, "set") == 0)
	{
		if (isFlasher())
		{
			if (from_socket) 
				Send2Socket("err flasher");
		}
		else if (strcmp(sub, "led") == 0)
		{
			char* status = strtok(NULL, " \r\n");
			if (status)
			{
				unsigned char nstat = (unsigned char)getno(status);

				setLedStatus(nstat);

				ok = true;
			}
		}
		else if (strcmp(sub, "lcd") == 0)
		{
			char* status = strtok(NULL, " \r\n");
			if (status)
			{
				unsigned char nstat = (unsigned char)getno(status);

				setLCDControl(nstat);

				ok = true;
			}
		}
		else if (strcmp(sub, "blight") == 0)
		{
			char* status = strtok(NULL, " \r\n");
			if (status)
			{
				unsigned char nstat = (unsigned char)getno(status);

				setBacklight(nstat);

				ok = true;
			}
		}
		else if (strcmp(sub, "contrast") == 0)
		{
			char* status = strtok(NULL, " \r\n");
			if (status)
			{
				unsigned char nstat = (unsigned char)getno(status);

				setContrast(nstat);

				ok = true;
			}
		}
		else if (strcmp(sub, "clear") == 0)
		{
			clearLCD();
			ok = true;
		}
		else if (strcmp(sub, "text") == 0)
		{
			char* x = strtok(NULL, " \r\n");
			char* y = strtok(NULL, " \r\n");
			char* text = strtok(NULL, "");
		    if (x && y && text)
			{
				char nx = (char)getno(x);
				if (nx < 0) nx = 0;
				if (nx > (LINE_LENGTH-1)) nx = LINE_LENGTH-1;

				char ny = (char)getno(y);
				if (ny < 0) ny = 0;
				if (ny > 1) ny = 1;

				int tl = convertslash(text);
				setText(nx, ny, text, tl);

				ok = true;
			}
		}
		else if (strcmp(sub, "font") == 0)
		{
			char* no = strtok(NULL, " \r\n");
			char* chars[8];
			chars[0] = strtok(NULL, " \r\n");
			chars[1] = strtok(NULL, " \r\n");
			chars[2] = strtok(NULL, " \r\n");
			chars[3] = strtok(NULL, " \r\n");
			chars[4] = strtok(NULL, " \r\n");
			chars[5] = strtok(NULL, " \r\n");
			chars[6] = strtok(NULL, " \r\n");
			chars[7] = strtok(NULL, " \r\n");

			if (no && chars[0] && chars[1] && chars[2] && chars[3] && chars[4] && chars[5] && chars[6] && chars[7])
			{
				char n = (char) getno(no);
				char c[8];
				c[0] = (char) getno(chars[0]);
				c[1] = (char) getno(chars[1]);
				c[2] = (char) getno(chars[2]);
				c[3] = (char) getno(chars[3]);
				c[4] = (char) getno(chars[4]);
				c[5] = (char) getno(chars[5]);
				c[6] = (char) getno(chars[6]);
				c[7] = (char) getno(chars[7]);

				setFont(n, c);

				ok = true;
			}
		}
		else if (strcmp(sub, "relay") == 0)
		{
			char* msec = strtok(NULL, " \r\n");
			long lmsec = atol(msec);
			setRelay(lmsec);
			ok = true;
		}
		else if (strcmp(sub, "splash") == 0)
		{
			testSplash();
			ok = true;
		}
		else if (strcmp(sub, "inte") == 0)
		{//external eprom
			if (writeFile4String("ieprom.hex", command + 9))//9 = set inte data
			{
				writeFile(from_socket, TYPE_EPROM_INTERNAL, "ieprom.hex", -1);
			}
			ok = true;
		}
		else if (strcmp(sub, "exte") == 0)
		{//external eprom
			if (writeFile4String("eeprom.hex", command + 9))//9 = set exte data
			{
				writeFile(from_socket, TYPE_EPROM_EXTERNAL, "eeprom.hex", -1);

				//write 5 to the ext splash screen counter
				HArray ha;

				ha.setHIDPkt(_getWriteMsg(TYPE_EPROM_INTERNAL), 3+1, 3, 0xff, 0, 1);
				ha.getBuf()[4] = 5;//5 splash screens

				m_bWriteOK = m_pParent->Write(&ha);
				int tmout = 0;
				while ((m_nError == -1)&&(tmout <= A_TIMEOUT))
				{
					Sleep(5);
					tmout++;
//					if (tmout > A_TIMEOUT) return -14;
				}
				//if (m_nError != 0) return m_nError;

				//if (!ha.equalsSpecial(&m_aReceived, 3+len))
				//	return -3;

				MyTrace(1, "write: addr: %x (%d)\n",0xFF, 1);
			}
			ok = true;
		}

	}//set
	else if (strcmp(cmd, "get") == 0)
	{
		if (isFlasher())
		{
			if (from_socket) 
				Send2Socket("err flasher");
		}
		else if (strcmp(sub, "inte") == 0)
		{//get eprom
			if (readFile(from_socket, TYPE_EPROM_INTERNAL, "ieprom_read.hex",0x0,5*BUF_LINE_LEN))//read only the screens
			{
				unsigned char data[5*BUF_LINE_LEN];
				if (loadFile2Array(data, 5*BUF_LINE_LEN, "ieprom_read.hex"))
				{
					char mesg[5*STR_LINE_LEN+1];
					char* message = mesg;
					memset(mesg, 0, sizeof(mesg));

					int nTime;
					
					for (int i=0;i<5;i++)
					{
						if (  (data[i*BUF_LINE_LEN] == 0xff)
							||(data[i*BUF_LINE_LEN+1] == 0xff)
							)
						{
							sprintf(message, "-1|-1|%02x|%02x|%02x|", data[i*BUF_LINE_LEN+2],data[i*BUF_LINE_LEN+3],data[i*BUF_LINE_LEN+4]);
						}
						else
						{
							nTime = data[i*BUF_LINE_LEN+1] & 0xff;
							nTime = nTime << 8;
							nTime = nTime | (data[i*BUF_LINE_LEN] & 0xff);
							sprintf(message, "%02d|%02d|%02x|%02x|%02x|",nTime / 60, nTime % 60, data[i*BUF_LINE_LEN+2],data[i*BUF_LINE_LEN+3],data[i*BUF_LINE_LEN+4]);
						}
						int j = 0;
						for (j=0;j<20;j++)
							message[15+j] = data[i*BUF_LINE_LEN+5+j];
						
						message[35] = '|';

						for (j=0;j<20;j++)
							message[36+j] = data[i*BUF_LINE_LEN+26+j];

						message[56] = '|';

						message = message + STR_LINE_LEN;
					}

					if (from_socket) 
						Send2Socket("get inte %s", mesg);
				}
			}
			ok = true;
		}
		else if (strcmp(sub, "exte") == 0)
		{//get eprom
			if (readFile(from_socket, TYPE_EPROM_EXTERNAL, "eeprom_read.hex",0x0,5*BUF_LINE_LEN))//read only the screens
			{
				unsigned char data[5*BUF_LINE_LEN];
				if (loadFile2Array(data, 5*BUF_LINE_LEN, "eeprom_read.hex"))
				{
					char mesg[5*STR_LINE_LEN+1];
					char* message = mesg;
					memset(mesg, 0, sizeof(mesg));

					int nTime;
					
					for (int i=0;i<5;i++)
					{
						if (  (data[i*BUF_LINE_LEN] == 0xff)
							||(data[i*BUF_LINE_LEN+1] == 0xff)
							)
						{
							sprintf(message, "-1|-1|%02x|%02x|%02x|", data[i*BUF_LINE_LEN+2],data[i*BUF_LINE_LEN+3],data[i*BUF_LINE_LEN+4]);
						}
						else
						{
							nTime = data[i*BUF_LINE_LEN+1] & 0xff;
							nTime = nTime << 8;
							nTime = nTime | (data[i*BUF_LINE_LEN] & 0xff);
							sprintf(message, "%02d|%02d|%02x|%02x|%02x|",nTime / 60, nTime % 60, data[i*BUF_LINE_LEN+2],data[i*BUF_LINE_LEN+3],data[i*BUF_LINE_LEN+4]);
						}
						int j = 0;
						for (j=0;j<20;j++)
							message[15+j] = data[i*BUF_LINE_LEN+5+j];
						
						message[35] = '|';

						for (j=0;j<20;j++)
							message[36+j] = data[i*BUF_LINE_LEN+26+j];

						message[56] = '|';

						message = message + STR_LINE_LEN;
					}

					if (from_socket) 
						Send2Socket("get exte %s", mesg);
				}
			}
			ok = true;
		}
	}
	
	m_lockCommand.unlock();

	if (!ok)
	{
		MyTrace(0, "ERROR: (Unknown command: %s)\n", command);
		if (!from_socket)
		{
			MyTrace(0, "Available commands:\n");
			MyTrace(0, " set led xff/12\n");
			MyTrace(0, " set blight 0/1\n");
			MyTrace(0, " set contrast 0/1\n");
			MyTrace(0, " set clear\n");
			MyTrace(0, " set text x y text\n");
			MyTrace(0, " set font no f0 f1 f2 f3 f4 f5 f6 f7\n");
			MyTrace(0, " set relay msec\n");
			MyTrace(0, " set splash\n");
			MyTrace(0, " set inte min|sec|nextslot|repetition|leds|line1(20chars)|line2(20chars)|min|sec|...\n");
			MyTrace(0, " set exte min|sec|nextslot|repetition|leds|line1(20chars)|line2(20chars)|min|sec|...\n");
			MyTrace(0, " get inte\n");
			MyTrace(0, " get exte\n");
		}
	}

	delete tcmd;
}

void HIDAcme::connectGraph(HIDGraphics* hg)
{
	m_lockCommand.lock();
	m_pGraphics = hg;
	m_lockCommand.unlock();
}

bool HIDAcme::loadFile2Array(unsigned char* data, const unsigned long len, const char* file)
{
	memset(data, 0xff, len);

	HEXFile hfile(file);
	if (hfile.Open() == 0)
	{
		unsigned long addr,oaddr = -1;
		int ret = hfile.NextLine();
		while (ret==0)
		{
			addr = hfile.GetGlobalAddress();
			for (unsigned long p = 0;p<hfile.GetLength();p++)
			{
				if ((addr+p)<len)
				{
					data[addr+p] = hfile.GetData()[p];
				}
				else 
					return true;
			}

			ret = hfile.NextLine();
			if (ret > 1) return false;
		}
	}
	else
	{
		MyTrace(1,"Invalid hex file!\n");

		return false;
	}

	return true;
}
