// HIDSocket.cpp: implementation of the HIDSocket class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "HIDSocket.h"
#include "HIDAcme.h"
#include "HIDClient.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////
class HGarbageCollector : public HThread
{
public:
	HGarbageCollector(HIDSocket* parent)
	{
		m_pParent = parent;
		start();
	}

	virtual ~HGarbageCollector()
	{
		stop();
	}

	virtual void run()
	{
		while (isRunning())
		{
			m_pParent->m_pClients->lock();
			ITEM *p = NULL;
			ITEM *t=m_pParent->m_pClients->getHead();
			while( t!=NULL )
			{
				HIDClient *hs=(HIDClient*) t->pdata;
				if (hs->isGarbage())
				{
					ITEM *tt = t->pnext;

					MyTrace(0, "   Garbage: deleting client %s\n", hs->getSocket()->GetIP());
					m_pParent->m_pClients->removeElement(p, t);
					delete hs;

					t = tt;
				}
				else
				{
					p = t;
					t=t->pnext;
				}
			}

			m_pParent->m_pClients->unlock();

			Sleep(250);
		}
	}

private:
	HIDSocket* m_pParent;
};

HIDSocket::HIDSocket(HIDAcme* device, int port)
{
	m_pClients = new HList();

	m_pDevice = device;

	m_pServer = new HServerSock(port);

	if (m_pServer->isRun())
		start();

	m_pDevice->SetSocket(this);

	m_pGarbColl = new HGarbageCollector(this);
}

HIDSocket::~HIDSocket()
{
	m_pGarbColl->stop();

	m_pServer->setStatus(-1);

	m_pClients->lock();
	ITEM *t=m_pClients->getHead();
	while( t!=NULL )
	{
		HIDClient *hs=(HIDClient*) t->pdata;
		hs->getSocket()->setStatus(-1);
		t=t->pnext;
	}
	m_pClients->unlock();
		
	stop();

	m_pClients->lock();
	t=m_pClients->getHead();
	while( t!=NULL )
	{
		HIDClient *hs=(HIDClient*) t->pdata;
		delete hs;
		t=t->pnext;
	}
	m_pClients->unlock();
	
	delete m_pServer;
	delete m_pGarbColl;
	delete m_pClients;
}

void HIDSocket::send2all(const char* msg)
{
	MyTrace(2, "S_SX:%s\n", msg);
	m_pClients->lock();
	ITEM* t=m_pClients->getHead();
	while( t!=NULL )
	{
		HIDClient *hs=(HIDClient*) t->pdata;
		
		send2one(hs->getSocket(), msg);

		t=t->pnext;
	}
	m_pClients->unlock();
}

void HIDSocket::send2one(HSocket* sock, const char* msg)
{
	sock->sendData(msg, strlen(msg));
	sock->sendData("\r\n", 2);
}

void HIDSocket::connected(HSocket* sock)
{
	if (m_pDevice)
	{
		char buf[256];buf[0]=0;
		m_pDevice->getinitmesgs(buf);
		MyTrace(2, "S_Sx:%s\n", buf);
		send2one(sock, buf); 
	}
}

void HIDSocket::run( void )
{
	char ipbuf[50];
	while (isRunning())
	{
		struct sockaddr_in sad;
		SOCKET sock = m_pServer->acceptSocket((sockaddr*)&sad );
		if (m_pServer->getStatus()!=-1)
		{
			sprintf(ipbuf, "%s", inet_ntoa( sad.sin_addr ));
			MyTrace(0, "CLIENT connected from IP %s\n", ipbuf );

			HSocket* socket = new HSocket( sock , ipbuf, m_pServer->getPort()); 
			socket->setMode();
			
			if (socket->getStatus() == 0)
			{
				connected(socket);

				HIDClient* pClient = new HIDClient(socket, this);
				m_pClients->add(pClient);
			}
			else delete socket;
		}

		Sleep(10);
	}
}

void HIDSocket::sendCmd(HIDClient* client, const char* cmd)
{
	m_pDevice->sendCmd(cmd, true);
}