#include "stdafx.h"
#include <stdlib.h>
#include "IRDecoder.h"

CIRDecoder::CIRDecoder(int type)
{
	m_nType = type;
	resetRC5();
}

CIRDecoder::~CIRDecoder(void)
{
}

int CIRDecoder::decodeIR(unsigned char* buff, int len)
{
	switch (m_nType)
	{
	case IR_RC5: return decodeRC5(buff, len);
	}

	return -1;
}

#define RC5_SHORT_PULSE		1
#define RC5_LONG_PULSE		2
#define RC5_SHORT_SPACE		3
#define RC5_LONG_SPACE		4

#define RC5_STATE_BEGIN		0
#define RC5_STATE_MID0		1
#define RC5_STATE_MID1		2
#define RC5_STATE_START0	3
#define RC5_STATE_START1	4

void CIRDecoder::resetRC5()
{
	m_rc5.nState = RC5_STATE_BEGIN;
	m_rc5.bBitsCount = 0;
	m_rc5.ulTimestamp = 0;
	memset(m_rc5.bBits, 0, 14);
}

int CIRDecoder::decodeRC5(unsigned char* buff, int len)
{
	unsigned long tstamp = GetTickCount();

	////printf(">>>%d\n",tstamp - m_rc5.ulTimestamp);
	if ((tstamp - m_rc5.ulTimestamp)>60)
	{
		//printf("IR_RC5: New command\n");
		resetRC5();
	}
	m_rc5.ulTimestamp = tstamp;

	if ((len % 2)!=0)
	{
		//printf("IR_RC5: Bad command\n");
		resetRC5();
		return -1;
	}

	int result = -1;

	//convert the buffer in short's (every pulse is on 2 bytes)
	short* buff_short = (short*)buff;

	for (int i=0;i<(len/2);i++)
	{
		if (!checkOneRc5Pulse(buff_short[i]))
		{
			//printf("IR_RC5: Bad pulse (%d=%d %x)\n",i,buff_short[i],buff_short[i]&0xFFFF);
			resetRC5();
		}
		else
		{
			////printf("IR_RC5: OK pulse (%d=%d %x)\n",i,buff_short[i],buff_short[i]&0xFFFF);
			if (m_rc5.bBitsCount >= 14)
			{
				result = 0;

				for (int b=0;b<14;b++)
				{
					if (m_rc5.bBits[b])	result |= 1;
					if (b!=13) result = result << 1;
				}
				
				resetRC5();
			}
		}
	}
	//printf("\n");

	return result;
}

void CIRDecoder::addRc5Bit(char bit)
{
//	//printf("(%d)", bit);
	m_rc5.bBits[m_rc5.bBitsCount] = bit;
	m_rc5.bBitsCount++;
}

bool CIRDecoder::checkOneRc5Pulse(short pulse)
{
	int   pulse_type = 0;

	pulse = -pulse;//negating the pulse (complementary)

	if (pulse < 0)
	{//space
		short abspulse = abs(pulse);

		if ( (abspulse > 444 ) && (abspulse < 1333) )
			 pulse_type = RC5_SHORT_SPACE;
		else if ( (abspulse > 1334 ) && (abspulse < 2222) )
			 pulse_type = RC5_LONG_SPACE;
	}
	else
	{//pulse
		if ( (pulse > 444 ) && (pulse < 1333) )
			 pulse_type = RC5_SHORT_PULSE;
		else if ( (pulse > 1334 ) && (pulse < 2222) )
			 pulse_type = RC5_LONG_PULSE;
	}

	if (pulse_type == 0) return false;//error

	if (m_rc5.nState == RC5_STATE_BEGIN)
	{
		m_rc5.nState = RC5_STATE_MID1;
		addRc5Bit(1);
		//printf("_|");
	}

	/*
	switch (pulse_type)
	{
	case RC5_SHORT_PULSE:	printf("~|");break;
	case RC5_LONG_PULSE:	printf("~~|");break;
	case RC5_SHORT_SPACE:	printf("_|");break;
	case RC5_LONG_SPACE:	printf("___|");break;
	}
	*/

	switch (m_rc5.nState)
	{
	case RC5_STATE_BEGIN:
		m_rc5.nState = RC5_STATE_MID1;
		addRc5Bit(1);
		break;

	case RC5_STATE_MID0:
		switch (pulse_type)
		{
		case RC5_SHORT_SPACE:
			m_rc5.nState = RC5_STATE_START0;
			break;
		case RC5_LONG_SPACE:
			m_rc5.nState = RC5_STATE_MID1;
			addRc5Bit(1);
			break;
		default:
			//printf("<RC5_STATE_MID0 Err (%d)>\n",pulse_type);
			return false;
		}
		break;

	case RC5_STATE_MID1:
		switch (pulse_type)
		{
		case RC5_SHORT_PULSE:
			m_rc5.nState = RC5_STATE_START1;
			break;
		case RC5_LONG_PULSE:
			m_rc5.nState = RC5_STATE_MID0;
			addRc5Bit(0);
			break;
		default:
			//printf("<RC5_STATE_MID1 Err (%d)>\n",pulse_type);
			return false;
		}
		break;

	case RC5_STATE_START0:
		switch (pulse_type)
		{
		case RC5_SHORT_PULSE:
			m_rc5.nState = RC5_STATE_MID0;
			addRc5Bit(0);
			break;
		default:
			//printf("<RC5_STATE_START0 Err (%d)>\n",pulse_type);
			return false;
		}
		break;

	case RC5_STATE_START1:
		switch (pulse_type)
		{
		case RC5_SHORT_SPACE:
			m_rc5.nState = RC5_STATE_MID1;
			addRc5Bit(1);
			break;
		default:
			//printf("<RC5_STATE_START1 Err (%d)>\n",pulse_type);
			return false;
		}
		break;
	
	default:
		return false;
	}

	return true;
}

		